import { Router } from 'express';
import emailService from '../lib/emailService.js';
const router = Router();
/**
 * @route   GET /email/test
 * @desc    Send a test email
 * @access  Admin only (should add auth middleware in production)
 */
router.get('/test', async (req, res) => {
    try {
        const { email } = req.query;
        // Validate email parameter
        if (!email || typeof email !== 'string') {
            return res.status(400).json({
                success: false,
                error: 'Valid email parameter is required',
            });
        }
        // Send test email
        const result = await emailService.sendTestEmail(email);
        if (result.success) {
            return res.status(200).json({
                success: true,
                message: `Test email sent successfully to ${email}`,
            });
        }
        else {
            return res.status(500).json({
                success: false,
                error: result.error || 'Failed to send email',
            });
        }
    }
    catch (error) {
        console.error('Test email error:', error);
        return res.status(500).json({
            success: false,
            error: error instanceof Error ? error.message : 'Failed to send test email',
        });
    }
});
/**
 * @route   GET /email/verify
 * @desc    Verify SMTP connection without sending an email
 * @access  Admin only
 */
router.get('/verify', async (req, res) => {
    try {
        // Attempt to verify the SMTP connection
        const result = await emailService.verifyConnection();
        if (result.success) {
            return res.status(200).json({
                success: true,
                message: 'SMTP connection verified successfully',
            });
        }
        else {
            return res.status(500).json({
                success: false,
                error: result.error || 'Failed to verify SMTP connection',
            });
        }
    }
    catch (error) {
        console.error('SMTP verification error:', error);
        return res.status(500).json({
            success: false,
            error: error instanceof Error ? error.message : 'Failed to verify SMTP connection',
        });
    }
});
/**
 * @route   GET /email/diagnostics
 * @desc    Check email configuration and troubleshoot issues
 * @access  Admin only
 */
router.get('/diagnostics', async (req, res) => {
    try {
        // Check if required environment variables are set
        const diagnostics = {
            environment: process.env.NODE_ENV || 'not set',
            smtpConfig: {
                host: process.env.SMTP_HOST || 'not set',
                port: process.env.SMTP_PORT || 'not set',
                secure: process.env.SMTP_SECURE === 'true',
                userConfigured: Boolean(process.env.SMTP_USER),
                passwordConfigured: Boolean(process.env.SMTP_PASS),
            },
            emailConfig: {
                fromName: process.env.FROM_NAME || 'not set',
                fromEmail: process.env.FROM_EMAIL || 'not set',
            },
            possibleIssues: [],
        };
        // Check for common issues
        if (!process.env.SMTP_HOST) {
            diagnostics.possibleIssues.push('SMTP_HOST environment variable is not set');
        }
        if (!process.env.SMTP_USER) {
            diagnostics.possibleIssues.push('SMTP_USER environment variable is not set');
        }
        if (!process.env.SMTP_PASS) {
            diagnostics.possibleIssues.push('SMTP_PASS environment variable is not set');
        }
        if (process.env.SMTP_HOST?.includes('gmail')) {
            diagnostics.possibleIssues.push('For Gmail, you might need to create an App Password instead of using your regular password. ' +
                'Visit https://myaccount.google.com/apppasswords');
        }
        // Send back the diagnostics
        return res.status(200).json({
            success: true,
            diagnostics,
            remediation: [
                'Check that all SMTP environment variables are correctly set',
                'For Gmail or Google Workspace accounts, create an App Password',
                'For Microsoft accounts, check if MFA is enabled and requires an App Password',
                'Verify that your SMTP provider allows authentication from your current IP',
                "Some providers require setting up 'Less secure app access' or similar settings",
            ],
        });
    }
    catch (error) {
        console.error('Email diagnostics error:', error);
        return res.status(500).json({
            success: false,
            error: error instanceof Error ? error.message : 'Failed to run email diagnostics',
        });
    }
});
/**
 * @route   GET /email/sent
 * @desc    Get a list of emails sent via stub transport
 * @access  Admin only (development mode only)
 */
router.get('/sent', (req, res) => {
    try {
        if (process.env.USE_LOCAL_EMAIL !== 'true') {
            return res.status(400).json({
                success: false,
                error: 'This endpoint is only available when USE_LOCAL_EMAIL=true',
            });
        }
        const sentEmails = emailService.getSentEmails();
        return res.status(200).json({
            success: true,
            count: sentEmails.length,
            emails: sentEmails,
        });
    }
    catch (error) {
        console.error('Error retrieving sent emails:', error);
        return res.status(500).json({
            success: false,
            error: error instanceof Error ? error.message : 'Failed to retrieve sent emails',
        });
    }
});
export default router;
//# sourceMappingURL=email.js.map